# Fichier de librairie.
# Classe de pilotage des écran LCD I2C

# Fonctionne avec :
# Grove - LCD RGB Backlight
# Sciencethic 651 018

from time import sleep

class LCD16x2RGB:

    # BackLight Adr & Reg
    BL_ADDR         = 0x62
    BL_MD1_REG      = 0x00
    BL_MD2_REG      = 0x01
    BL_BLUE_REG     = 0x02
    BL_GREEN_REG    = 0x03
    BL_RED_REG      = 0x04
    BL_OUT_REG      = 0x08

    # LCD Adr & Reg
    LCD_ADDR = 0x3E
    LCD_CLEARDISPLAY = 0x01
    LCD_RETURNHOME = 0x02
    LCD_ENTRYMODESET = 0x04
    LCD_DISPLAYCONTROL = 0x08
    LCD_FUNCTIONSET = 0x20
    LCD_ENTRYLEFT = 0x02
    LCD_ENTRYSHIFTINCREMENT = 0x01
    LCD_ENTRYSHIFTDECREMENT = 0x00
    LCD_DISPLAYON = 0x04
    LCD_CURSORON = 0x02
    LCD_CURSOROFF = 0x00
    LCD_BLINKON = 0x01
    LCD_BLINKOFF = 0x00
    LCD_2LINE = 0x08
    
    def __init__(self,plugUno):
        self.plugUno = plugUno

        # Activate the I2C peripherals
        plugUno.i2c_configure()

        # 16X2 I2C controller Init
        self.disp_func = self.LCD_DISPLAYON | self.LCD_2LINE
        self.plugUno.i2c_write(self.LCD_ADDR,[0x80,self.LCD_FUNCTIONSET | self.disp_func])
        sleep(0.0045)
        self.plugUno.i2c_write(self.LCD_ADDR,[0x80,self.LCD_FUNCTIONSET | self.disp_func])
        sleep(0.000150)
        self.plugUno.i2c_write(self.LCD_ADDR,[0x80,self.LCD_FUNCTIONSET | self.disp_func])
        self.disp_ctrl = self.LCD_DISPLAYON | self.LCD_CURSOROFF | self.LCD_BLINKOFF
        self.display(True)
        self.clear()    
        self.disp_mode = self.LCD_ENTRYLEFT | self.LCD_ENTRYSHIFTDECREMENT
        self.plugUno.i2c_write(self.LCD_ADDR,[0x80,self.LCD_ENTRYMODESET | self.disp_mode])

        # LCD Backlight init
        self.plugUno.i2c_write(self.BL_ADDR,[self.BL_MD1_REG,0x00])
        self.plugUno.i2c_write(self.BL_ADDR,[self.BL_MD2_REG,0x00])
        self.plugUno.i2c_write(self.BL_ADDR,[self.BL_OUT_REG,0xAA])

        sleep(0.010)
        self.setColor(255,255,255)


    def writeReg(self,devAdr,regAdr,val):
        self.plugUno.i2c_write(devAdr,[regAdr,val])

    def print(self, text):
        for char in text:
            self.plugUno.i2c_write(self.LCD_ADDR,[0x40,ord(char)])

    def cursor(self, state):
        if state:
            self.disp_ctrl |= self.LCD_CURSORON
            self.writeReg(self.LCD_ADDR,0x80,self.LCD_DISPLAYCONTROL  | self.disp_ctrl)
        else:
            self.disp_ctrl &= ~self.LCD_CURSORON
            self.writeReg(self.LCD_ADDR,0x80,self.LCD_DISPLAYCONTROL  | self.disp_ctrl)

    def setCursor(self, col, row):
        col = (col | 0x80) if row == 0 else (col | 0xc0)
        self.writeReg(self.LCD_ADDR,0x80,col)

    def blink(self, state):
        if state:
            self.disp_ctrl |= self.LCD_BLINKON
            self.writeReg(self.LCD_ADDR,0x80,self.LCD_DISPLAYCONTROL  | self.disp_ctrl)
        else:
            self.disp_ctrl &= ~self.LCD_BLINKON
            self.writeReg(self.LCD_ADDR,0x80,self.LCD_DISPLAYCONTROL  | self.disp_ctrl)

    def display(self, state):
        if state:
            self.disp_ctrl |= self.LCD_DISPLAYON
            self.writeReg(self.LCD_ADDR,0x80,self.LCD_DISPLAYCONTROL  | self.disp_ctrl)
        else:
            self.disp_ctrl &= ~self.LCD_DISPLAYON
            self.writeReg(self.LCD_ADDR,0x80,self.LCD_DISPLAYCONTROL  | self.disp_ctrl)

    def clear(self):
        self.writeReg(self.LCD_ADDR,0x80,self.LCD_CLEARDISPLAY)
        sleep(0.002)

    def home(self):
        self.writeReg(self.LCD_ADDR,0x80,self.LCD_RETURNHOME)
        sleep(0.002)

    def setColor(self,r,g,b):
        self.writeReg(self.BL_ADDR,self.BL_RED_REG,r)
        self.writeReg(self.BL_ADDR,self.BL_GREEN_REG,g)
        self.writeReg(self.BL_ADDR,self.BL_BLUE_REG,b)
    
